import React, { useState, useEffect } from 'react';
import axios from 'axios';
import "./mediaquery.css"; // Import the CSS file
import * as XLSX from 'xlsx';

const App = () => {
    const [ips, setIps] = useState([]);
    const [newIp, setNewIp] = useState('');
    const [newTitle, setNewTitle] = useState('');
    const [newContactPersonName, setNewContactPersonName] = useState('');
    const [newContactPersonNumber, setNewContactPersonNumber] = useState('');
    const [newLocation, setNewLocation] = useState('');
    const [currentTime, setCurrentTime] = useState(new Date());
    const [lastRefreshed, setLastRefreshed] = useState(new Date());
    const [showAddPopup, setShowAddPopup] = useState(false);
    const [showManagePopup, setShowManagePopup] = useState(false);
    const [showEditPopup, setShowEditPopup] = useState(false);
    const [selectedIp, setSelectedIp] = useState(null);
    const [selectedTitle, setSelectedTitle] = useState('');
    const [selectedContactPersonName, setSelectedContactPersonName] = useState('');
    const [selectedContactPersonNumber, setSelectedContactPersonNumber] = useState('');
    const [successMessage, setSuccessMessage] = useState('');
    const [errorMessage, setErrorMessage] = useState('');
    const [searchTerm, setSearchTerm] = useState('');
    const [historyData, setHistoryData] = useState([]);
    const [showHistoryPopup, setShowHistoryPopup] = useState(false);

    const API_BASE_URL = 'https://back.onlinefashionbd.com'; // Replace with your backend URL

    useEffect(() => {
        fetchIps();
        const interval = setInterval(() => {
            fetchIps();
            setLastRefreshed(new Date());
        }, 5000);
        return () => clearInterval(interval);
    }, []);

    useEffect(() => {
        const clockInterval = setInterval(() => {
            setCurrentTime(new Date());
        }, 1000);
        return () => clearInterval(clockInterval);
    }, []);

    const fetchIps = async () => {
        const response = await axios.get(`${API_BASE_URL}/ips`);
        setIps(response.data);
    };

    const addIp = async () => {
        if (!isValidIp(newIp)) {
            setErrorMessage('Invalid IP Address! Please enter a valid IP.');
            setTimeout(() => setErrorMessage(''), 3000);
            return;
        }

        if (newIp && newTitle && newLocation) {
            try {
                const response = await axios.post(`${API_BASE_URL}/ips`, {
                    ip: newIp,
                    title: newTitle,
                    location: newLocation,
                    contactPersonName: newContactPersonName,
                    contactPersonNumber: newContactPersonNumber,
                });
                if (response.data) {
                    setIps(response.data);
                    setNewIp('');
                    setNewTitle('');
                    setNewLocation('');
                    setNewContactPersonName('');
                    setNewContactPersonNumber('');
                    setShowAddPopup(false);
                    setSuccessMessage('IP Address added successfully!');
                    setTimeout(() => setSuccessMessage(''), 3000);
                } else {
                    setErrorMessage('Failed to add IP Address!');
                    setTimeout(() => setErrorMessage(''), 3000);
                }
            } catch (error) {
                setErrorMessage('Failed to add IP Address!');
                setTimeout(() => setErrorMessage(''), 3000);
            }
        }
    };

    const isValidIp = (ip) => {
        const regex = /^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
        return regex.test(ip);
    };

    const deleteIp = async (ip) => {
        await axios.delete(`${API_BASE_URL}/ips/${ip}`);
        fetchIps();
    };

    const updateIp = async () => {
        if (selectedIp) {
            try {
                const response = await axios.put(`${API_BASE_URL}/ips/${selectedIp.ip}`, {
                    title: newTitle,
                    location: newLocation,
                    contactPersonName: newContactPersonName,
                    contactPersonNumber: newContactPersonNumber,
                });
                if (response.data && response.data.success) {
                    setIps((prevIps) =>
                        prevIps.map((ip) =>
                            ip.ip === selectedIp.ip
                                ? { ...ip, title: newTitle, location: newLocation, contactPersonName: newContactPersonName, contactPersonNumber: newContactPersonNumber }
                                : ip
                        )
                    );
                    setShowEditPopup(false);
                    setSuccessMessage('IP Address updated successfully!');
                    setTimeout(() => setSuccessMessage(''), 3000);
                } else {
                    throw new Error('Failed to update IP Address!');
                }
            } catch (error) {
                console.error(error);
                setErrorMessage('Failed to update IP Address!');
                setTimeout(() => setErrorMessage(''), 3000);
            }
        }
    };

    const formatDate = (dateString) => {
        if (!dateString) return '-';
        const date = new Date(dateString);
        if (isNaN(date.getTime())) return '-';
        const day = date.getDate();
        const month = date.toLocaleString('default', { month: 'short' });
        const year = date.getFullYear();
        let hours = date.getHours();
        let minutes = date.getMinutes();
        let seconds = date.getSeconds();
        const ampm = hours >= 12 ? 'PM' : 'AM';
        hours = hours % 12 || 12;
        minutes = minutes < 10 ? `0${minutes}` : minutes;
        seconds = seconds < 10 ? `0${seconds}` : seconds;
        const formattedHours = hours < 10 ? `0${hours}` : hours;
        return `${day}-${month}-${year}, ${formattedHours}:${minutes}:${seconds} ${ampm}`;
    };

    const filteredIps = ips.filter(ip =>
        ip.ip.toLowerCase().includes(searchTerm.toLowerCase()) ||
        (ip.title && ip.title.toLowerCase().includes(searchTerm.toLowerCase())) ||
        (ip.location && ip.location.toLowerCase().includes(searchTerm.toLowerCase()))
    );

    const fetchHistory = async (ip) => {
        try {
            const response = await axios.get(`${API_BASE_URL}/ips/${ip}/history`);
            setHistoryData(response.data);
        } catch (error) {
            console.error('Error fetching IP history:', error);
        }
    };

    const handleIpClick = (ip, title, contactPersonName, contactPersonNumber) => {
        setSelectedIp(ip);
        setSelectedTitle(title);
        setSelectedContactPersonName(contactPersonName);
        setSelectedContactPersonNumber(contactPersonNumber);
        fetchHistory(ip);
        setShowHistoryPopup(true);
    };

    const formatDuration = (seconds) => {
        if (seconds === null || seconds === undefined) return '-';
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = Math.floor(seconds % 60);
        let duration = '';
        if (hours > 0) duration += `${hours}h `;
        if (minutes > 0) duration += `${minutes}m `;
        if (secs > 0 || duration === '') duration += `${secs}s`;
        return duration.trim();
    };

    const handleFileUpload = (event) => {
        const file = event.target.files[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = (e) => {
            const data = new Uint8Array(e.target.result);
            const workbook = XLSX.read(data, { type: 'array' });
            const sheetName = workbook.SheetNames[0];
            const worksheet = workbook.Sheets[sheetName];
            const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1 });

            const ipsToAdd = jsonData.slice(1).map((row) => ({
                ip: row[0],
                title: row[1],
                location: row[2],
                contactPersonName: row[3],
                contactPersonNumber: row[4],
            }));

            addIpsFromExcel(ipsToAdd);
        };
        reader.readAsArrayBuffer(file);
    };

    const addIpsFromExcel = async (ips) => {
        try {
            const response = await axios.post(`${API_BASE_URL}/ips/bulk`, ips);
            if (response.data) {
                setIps(response.data);
                setSuccessMessage('IPs added successfully from Excel file!');
                setTimeout(() => setSuccessMessage(''), 3000);
            }
        } catch (error) {
            setErrorMessage('Failed to add IPs from Excel file!');
            setTimeout(() => setErrorMessage(''), 3000);
        }
    };

    return (
        <div className="app-container">
            <img 
                src="/TNA.jpg" 
                alt="Logo" 
                className="logo"
            />
            <div className="time-display">
                <div className="current-time">Current Time: {formatDate(currentTime)}</div>
                <div className="last-refreshed">Last Refreshed: {formatDate(lastRefreshed)}</div>
            </div>
            <h1>Network Monitoring Tool</h1>
            <div className="controls">
                <button onClick={() => setShowAddPopup(true)}>Add IP</button>
                <button onClick={() => setShowManagePopup(true)}>Manage IP</button>
                <input
                    type="text"
                    placeholder="Search IP, Title, or Location"
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                />
            </div>

            {showAddPopup && (
                <div className="popup">
                    <h2>Add IP Address</h2>
                    <div className="form-group">
                        <label>IP Address:</label>
                        <input
                            type="text"
                            placeholder="Enter IP address"
                            value={newIp}
                            onChange={(e) => setNewIp(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Title:</label>
                        <input
                            type="text"
                            placeholder="Enter Title"
                            value={newTitle}
                            onChange={(e) => setNewTitle(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Location:</label>
                        <input
                            type="text"
                            placeholder="Enter Location"
                            value={newLocation}
                            onChange={(e) => setNewLocation(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Contact Person Name:</label>
                        <input
                            type="text"
                            placeholder="Enter Contact Person Name"
                            value={newContactPersonName}
                            onChange={(e) => setNewContactPersonName(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Contact Person Number:</label>
                        <input
                            type="number"
                            placeholder="Enter Contact Person Number"
                            value={newContactPersonNumber}
                            onChange={(e) => setNewContactPersonNumber(e.target.value)}
                        />
                    </div>
                    <div className="search">
                        <label>Upload Excel File:</label>
                        <input
                            type="file"
                            accept=".xlsx, .xls"
                            onChange={handleFileUpload}
                        />
                    </div>
                    <div className="form-actions">
                        <button onClick={addIp}                             
                        style={{
                                padding: '5px 10px',
                                marginRight: '10px',
                                backgroundColor: 'green',
                                color: 'white',
                                border: 'none',
                                borderRadius: '4px',
                                cursor: 'pointer',
                            }}>Save</button>
                        <button onClick={() => setShowAddPopup(false)}                            
                        style={{
                                padding: '5px 10px',
                                backgroundColor: 'red',
                                color: 'white',
                                border: 'none',
                                borderRadius: '4px',
                                cursor: 'pointer',
                            }}>Cancel</button>
                    </div>
                </div>
            )}

            {showManagePopup && (
                <div className="popup">
                    <button className="close-btn" onClick={() => setShowManagePopup(false)}>Close</button>
                    <h2>Manage IP Addresses</h2>
                    <div className="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>IP Address</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                {ips.map((item, index) => (
                                    <tr key={index}>
                                        <td>{item.ip}</td>
                                        <td>
                                            <span onClick={() => {
                                                setSelectedIp(item);
                                                setNewTitle(item.title || '');
                                                setNewLocation(item.location || '');
                                                setNewContactPersonName(item.contactPersonName || '');
                                                setNewContactPersonNumber(item.contactPersonNumber || '');
                                                setShowEditPopup(true);
                                            }}>✏️</span>
                                            <span onClick={() => deleteIp(item.ip)}>🗑️</span>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            )}

            {showEditPopup && selectedIp && (
                <div className="popup">
                    <h2>Edit IP Address</h2>
                    <div className="form-group">
                        <label>Title:</label>
                        <input
                            type="text"
                            placeholder="Enter Title"
                            value={newTitle}
                            onChange={(e) => setNewTitle(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Location:</label>
                        <input
                            type="text"
                            placeholder="Enter Location"
                            value={newLocation}
                            onChange={(e) => setNewLocation(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Contact Person Name:</label>
                        <input
                            type="text"
                            placeholder="Enter Contact Person Name"
                            value={newContactPersonName}
                            onChange={(e) => setNewContactPersonName(e.target.value)}
                        />
                    </div>
                    <div className="form-group">
                        <label>Contact Person Number:</label>
                        <input
                            type="text"
                            placeholder="Enter Contact Person Number"
                            value={newContactPersonNumber}
                            onChange={(e) => setNewContactPersonNumber(e.target.value)}
                        />
                    </div>
                    <div className="form-actions">
                        <button onClick={updateIp}                             
                        style={{
                                padding: '5px 10px',
                                marginRight: '10px',
                                backgroundColor: 'green',
                                color: 'white',
                                border: 'none',
                                borderRadius: '4px',
                                cursor: 'pointer',
                            }}>Update</button>
                        <button onClick={() => setShowEditPopup(false)}                            
                        style={{
                                padding: '5px 10px',
                                backgroundColor: 'red',
                                color: 'white',
                                border: 'none',
                                borderRadius: '4px',
                                cursor: 'pointer',
                            }}>Cancel</button>
                    </div>
                </div>
            )}

            {successMessage && (
                <div className="message success">
                    {successMessage}
                </div>
            )}

            {errorMessage && (
                <div className="message error">
                    {errorMessage}
                </div>
            )}

            <div className="main-table">
                <table>
                    <thead>
                        <tr>
                            <th>IP Address</th>
                            <th>Title</th>
                            <th>Location</th>
                            <th>Status</th>
                            <th>Last Downtime</th>
                            <th>Last Uptime</th>
                            <th>Last Status Change</th>
                            <th>Total Downtime</th>
                            <th>Total Uptime</th>
                            <th>IP Monitoring Start Time</th>
                        </tr>
                    </thead>
                    <tbody>
                        {filteredIps.map((item, index) => (
                            <tr key={index}>
                                <td>
                                    <span onClick={() => handleIpClick(item.ip, item.title, item.contactPersonName, item.contactPersonNumber)}>
                                        {item.ip}
                                    </span>
                                </td>
                                <td>{item.title || 'N/A'}</td>
                                <td>{item.location || 'N/A'}</td>
                                <td style={{ color: item.status === 'online' ? 'green' : 'red' }}>
                                    {item.status}
                                </td>
                                <td>{item.lastFailedOn ? formatDate(item.lastFailedOn) : 'N/A'}</td>
                                <td>{item.lastSucceedOn ? formatDate(item.lastSucceedOn) : 'N/A'}</td>
                                <td>{item.lastStatusChange ? formatDate(item.lastStatusChange) : 'N/A'}</td>
                                <td>{item.totalDowntimeFormatted || '0h 0m'}</td>
                                <td>{item.totalUptimeFormatted || '0h 0m'}</td>
                                <td>{item.startTime ? formatDate(item.startTime) : 'N/A'}</td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>

            {showHistoryPopup && (
                <div className="popup history-popup">
                    <div className="history-header">
                    <div className="history-actions">
                            <button onClick={() => {
                                const newWindow = window.open('', '_blank', 'width=600,height=400');
                                newWindow.document.write(`
                                    <html>
                                        <head>
                                            <title>IP History</title>
                                            <style>
                                                table {
                                                    width: 100%;
                                                    margin-top: 20px;
                                                    border-collapse: collapse;
                                                    text-align: center;
                                                    border: 1px solid #ddd;
                                                }
                                                th, td {
                                                    padding: 10px;
                                                    border: 1px solid #ddd;
                                                }
                                            </style>
                                        </head>
                                        <body>
                                            <h2>IP History</h2>
                                            <p>IP Address: ${selectedIp}</p>
                                            <p>Title: ${selectedTitle}</p>
                                            <p>Contact Person: ${selectedContactPersonName}</p>
                                            <p>Contact Number: ${selectedContactPersonNumber}</p>
                                            <table>
                                                <thead>
                                                    <tr>
                                                        <th>Status</th>
                                                        <th>Started Time</th>
                                                        <th>Ended Time</th>
                                                        <th>Duration</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    ${historyData.length > 0 ? 
                                                        historyData.map(history => `
                                                            <tr>
                                                                <td>${history.status}</td>
                                                                <td>${formatDate(history.startedAt)}</td>
                                                                <td>${formatDate(history.endedAt)}</td>
                                                                <td>${formatDuration(history.duration)}</td>
                                                            </tr>
                                                        `).join('')
                                                        : `
                                                        <tr>
                                                            <td colspan="4" style="text-align: center; color: #999;">No data available</td>
                                                        </tr>
                                                    `}
                                                </tbody>
                                            </table>
                                        </body>
                                    </html>
                                `);
                                newWindow.document.close();
                            }}                             
                            style={{
                                    padding: '5px 10px',
                                    marginRight: '10px',
                                    backgroundColor: 'green',
                                    color: 'white',
                                    border: 'none',
                                    borderRadius: '4px',
                                    cursor: 'pointer',
                                }}>
                                Open in New Window
                            </button>
                            <button onClick={() => setShowHistoryPopup(false)}                            
                            style={{
                                padding: '5px 10px',
                                backgroundColor: 'red',
                                color: 'white',
                                border: 'none',
                                borderRadius: '4px',
                                cursor: 'pointer',
                            }}>
                                Close
                            </button>
                        </div>
                        <div className='history-text'>
                            <h2>IP History</h2>
                            <p>IP Address: {selectedIp}</p>
                            <p>Title: {selectedTitle}</p>
                            <p>Contact Person: {selectedContactPersonName}</p>
                            <p>Contact Number: {selectedContactPersonNumber}</p>
                        </div>
                    </div>
                    <div className="history-table">
                        <table>
                            <thead>
                                <tr>
                                    <th>Status</th>
                                    <th>Started Time</th>
                                    <th>Ended Time</th>
                                    <th>Duration</th>
                                </tr>
                            </thead>
                            <tbody>
                                {historyData.length > 0 ? (
                                    historyData.map((history, index) => (
                                        <tr key={index}>
                                            <td>{history.status}</td>
                                            <td>{formatDate(history.startedAt)}</td>
                                            <td>{formatDate(history.endedAt)}</td>
                                            <td>{formatDuration(history.duration)}</td>
                                        </tr>
                                    ))
                                ) : (
                                    <tr>
                                        <td colSpan="4">No data available</td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            )}

            <footer>
                &copy; {new Date().getFullYear()} Network Monitoring Tool - All Rights Reserved By ATP-IT || Developed By Nizhum
            </footer>
        </div>
    );
};

export default App;